-- FILE: 003_story_state.sql
-- PURPOSE: Enhanced story state storage for visual novel system
-- DATABASE: SQLite (embedded in Flutter client)
-- AI_CONTEXT: This migration adds comprehensive story state tracking for Phase S6.
--             Replaces the simple chapter-based story_progress table with a more
--             flexible JSON-based system that supports:
--             - Episode progress with multiple status types
--             - Story flags for branching decisions
--             - Battle beat tracking (one-shot triggers)
--             - Current position save/resume
--             - Play count and completion timestamps

-- Migration: 003_story_state
-- Created: 2025-12-12
-- Phase: 2G (Story Mode)

-- ============================================================================
-- 1. STORY STATE TABLE
-- ============================================================================
-- Single row storing complete story state as JSON
-- This approach is simpler than multiple tables for a single-player save

CREATE TABLE IF NOT EXISTS story_state (
    state_id INTEGER PRIMARY KEY DEFAULT 1,  -- Only one row, always ID 1
    
    -- Complete state as JSON blob
    state_data TEXT NOT NULL,
    
    -- Quick-access fields for UI (duplicated from JSON for performance)
    current_episode_id TEXT,
    total_episodes_completed INTEGER DEFAULT 0,
    total_play_time_seconds INTEGER DEFAULT 0,
    
    -- Timestamps
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    
    -- Enforce single row
    CHECK (state_id = 1)
);

-- Insert default empty state
INSERT OR IGNORE INTO story_state (state_id, state_data, created_at, updated_at)
VALUES (1, '{}', datetime('now'), datetime('now'));

-- ============================================================================
-- 2. STORY SAVE SLOTS TABLE (Future: Multiple save files)
-- ============================================================================
-- For future support of multiple save slots

CREATE TABLE IF NOT EXISTS story_save_slots (
    slot_id INTEGER PRIMARY KEY,
    slot_name TEXT NOT NULL DEFAULT 'Save 1',
    
    -- State JSON blob
    state_data TEXT NOT NULL,
    
    -- Quick access
    current_episode_id TEXT,
    total_episodes_completed INTEGER DEFAULT 0,
    
    -- Timestamps
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    last_played_at TEXT NOT NULL
);

-- ============================================================================
-- 3. STORY ACHIEVEMENTS TABLE
-- ============================================================================
-- Track story-related achievements

CREATE TABLE IF NOT EXISTS story_achievements (
    achievement_id TEXT PRIMARY KEY,
    achievement_name TEXT NOT NULL,
    achievement_description TEXT,
    
    -- Progress
    is_unlocked INTEGER DEFAULT 0,
    progress_current INTEGER DEFAULT 0,
    progress_required INTEGER DEFAULT 1,
    
    -- Metadata
    unlocked_at TEXT,
    episode_id TEXT,  -- If tied to specific episode
    
    -- Icon/display
    icon_name TEXT
);

-- Pre-populate some achievements
INSERT OR IGNORE INTO story_achievements (achievement_id, achievement_name, achievement_description, progress_required) VALUES
    ('first_victory', 'First Victory', 'Win your first story battle', 1),
    ('tutorial_complete', 'Academy Student', 'Complete the tutorial episode', 1),
    ('fool_defeated', 'Fool''s Errand', 'Defeat The Fool', 1),
    ('perfect_victory', 'Flawless', 'Win a battle without losing your King any HP', 1),
    ('story_chapter_1', 'Chapter 1 Complete', 'Complete all Chapter 1 episodes', 1),
    ('all_choices', 'Path Walker', 'See all choice branches in an episode', 1),
    ('speedrunner', 'Speed Reader', 'Complete an episode in under 5 minutes', 1),
    ('collector', 'Lore Hunter', 'Unlock all story gallery entries', 1);

-- ============================================================================
-- 4. STORY GALLERY TABLE
-- ============================================================================
-- Unlocked gallery entries (character bios, artwork, music)

CREATE TABLE IF NOT EXISTS story_gallery (
    gallery_id TEXT PRIMARY KEY,
    gallery_type TEXT NOT NULL,  -- 'character', 'artwork', 'music', 'scene'
    gallery_name TEXT NOT NULL,
    gallery_description TEXT,
    
    -- Unlock status
    is_unlocked INTEGER DEFAULT 0,
    unlocked_at TEXT,
    unlock_episode_id TEXT,
    
    -- Content reference
    asset_path TEXT,
    
    -- Display order
    sort_order INTEGER DEFAULT 0
);

-- Pre-populate gallery entries for main characters
INSERT OR IGNORE INTO story_gallery (gallery_id, gallery_type, gallery_name, gallery_description, sort_order) VALUES
    ('char_empress', 'character', 'The Empress', 'Headmistress of the Astral Academy. A powerful soul who guides new students.', 1),
    ('char_eris', 'character', 'Eris', 'Co-headmistress and mentor. Her mysterious past holds many secrets.', 2),
    ('char_fool', 'character', 'The Fool', 'The first Major Arcana you face. Represents new beginnings and unlimited potential.', 3),
    ('char_magician', 'character', 'The Magician', 'A cunning strategist among the Major Arcana.', 4),
    ('char_priestess', 'character', 'The High Priestess', 'Keeper of secrets and hidden knowledge.', 5);

-- ============================================================================
-- END OF MIGRATION 003
-- ============================================================================
